//
//  EZOnlineCameraCell.m
//  EZOpenSDKDemo
//
//  Created by yuqian on 2020/3/10.
//  Copyright © 2020 hikvision. All rights reserved.
//

#import "EZOnlineCameraCell.h"
#import "EZPlayer.h"
#import "HIKLoadView.h"

@interface EZOnlineCameraCell() <EZPlayerDelegate>

@property (nonatomic, strong) UIView *playView;
@property (nonatomic, strong) HIKLoadView *loadingView;
@property (nonatomic, strong) UILabel *messageLabel;
@property (nonatomic, strong) UIButton *playerPlayButton;

@property (nonatomic, strong) EZPlayer *player;
@property (nonatomic, copy) NSString *deviceSerial;
@property (nonatomic, assign) NSInteger cameraNo;
@property (nonatomic, copy) NSString *verifyCode;

@end


@implementation EZOnlineCameraCell

- (instancetype)initWithFrame:(CGRect)frame
{
    self = [super initWithFrame:frame];
    if (self) {
        [self layoutUI];
    }
    return self;
}

- (void)startPlayWithDeviceSerial:(NSString *)deviceSerial cameraNo:(NSInteger)cameraNo verifyCode:(NSString *)code {
    if (deviceSerial.length == 0 ) {
        return;
    }
    
    self.deviceSerial = deviceSerial;
    self.cameraNo = cameraNo;
    self.verifyCode = [NSString stringWithFormat:@"%@", code];
    
    [self startPlay];
}
 
- (void)startPlay {
    if (_isPlaying) {
        [self stopPlay];
    }
    
    self.isPlaying = YES;
    
    _player = [EZOPENSDK createPlayerWithDeviceSerial:self.deviceSerial cameraNo:self.cameraNo useSubStream:YES];
    _player.delegate = self;
    
    if (self.verifyCode.length > 0) {
        [_player setPlayVerifyCode:self.verifyCode];
    }
    
    [_player setPlayerView:_playView];
    [_player startRealPlay];
    
    _playerPlayButton.hidden = YES;
    _loadingView.hidden = NO;
    [self.loadingView startSquareClcokwiseAnimation];
}

- (void)stopPlay {
    self.isPlaying = NO;
    _playerPlayButton.hidden = NO;
    [self.loadingView stopSquareClockwiseAnimation];
    
    [_player stopRealPlay];
}

- (void)setPlayerVerifyCode:(NSString *)code {
    [_player setPlayVerifyCode:code];
}

#pragma mark - PlayerDelegate Methods
- (void)player:(EZPlayer *)player didPlayFailed:(NSError *)error {
    NSLog(@"player: %@, didPlayFailed: %@", player, error);
    //如果是需要验证码或者是验证码错误
    NSString *tip = NSLocalizedString(@"device_play_fail", @"播放失败");
    if (error.code == EZ_SDK_NEED_VALIDATECODE
//        || error.code == EZ_SDK_VALIDATECODE_NOT_MATCH
        ) {

        tip = @"设备验证码错误";
    }
    
    [_player stopRealPlay];
    
    self.isPlaying = NO;
    _playerPlayButton.hidden = NO;
    
    [self.loadingView stopSquareClockwiseAnimation];
    
    self.messageLabel.text = [NSString stringWithFormat:@"%@(%d)",tip, (int)error.code];
    self.messageLabel.hidden = NO;
}

- (void)player:(EZPlayer *)player didReceivedMessage:(NSInteger)messageCode {
    NSLog(@"player: %@, didReceivedMessage: %d", player, (int)messageCode);
    if (messageCode == PLAYER_REALPLAY_START) {
        [self.loadingView stopSquareClockwiseAnimation];
        self.messageLabel.hidden = YES;

        [_player closeSound];
        NSLog(@"GetInnerPlayerPort:%d", [self.player getInnerPlayerPort]);
        NSLog(@"GetStreamFetchType:%d", [self.player getStreamFetchType]);
    } else if (messageCode == PLAYER_NET_CHANGED) {
        [_player stopRealPlay];
        [_player startRealPlay];
    }
}

#pragma mark - UI

- (void)layoutUI {
    [self.contentView addSubview:self.playView];
    [self.playView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.edges.equalTo(self.contentView);
    }];
    
    [self.contentView addSubview:self.loadingView];
    [self.loadingView mas_makeConstraints:^(MASConstraintMaker *make) {
        make.width.height.equalTo(@14);
        make.center.equalTo(self.playView);
    }];
    
    [self.contentView addSubview:self.messageLabel];
    [self.messageLabel mas_makeConstraints:^(MASConstraintMaker *make) {
        make.center.equalTo(self.contentView);
        make.left.right.equalTo(self.contentView);
    }];
    
    [self.contentView addSubview:self.playerPlayButton];
    [self.playerPlayButton mas_makeConstraints:^(MASConstraintMaker *make) {
        make.center.equalTo(self.contentView);
        make.width.height.equalTo(@54);
    }];
}

- (UIView *)playView {
    if (!_playView) {
        _playView = [[UIView alloc]init];
        _playView.backgroundColor = [UIColor blackColor];
        _playView.layer.masksToBounds = YES;
        _playView.layer.cornerRadius = 10.0;
    }
    return _playView;
}

- (HIKLoadView *)loadingView {
    if (!_loadingView) {
        _loadingView = [[HIKLoadView alloc] initWithHIKLoadViewStyle:HIKLoadViewStyleSqureClockWise];
        _loadingView.hidden = YES;
    }
    return _loadingView;
}

- (UIButton *)playerPlayButton {
    if (!_playerPlayButton) {
        _playerPlayButton = [UIButton buttonWithType:UIButtonTypeCustom];
        [_playerPlayButton setImage:[UIImage imageNamed:@"play"] forState:UIControlStateNormal];
        [_playerPlayButton addTarget:self action:@selector(startPlay) forControlEvents:UIControlEventTouchUpInside];
    }
    return _playerPlayButton;
}

- (UILabel *)messageLabel {
    if (!_messageLabel) {
        _messageLabel = [[UILabel alloc]init];
        _messageLabel.font = [UIFont systemFontOfSize:12.0];
        _messageLabel.textColor = [UIColor whiteColor];
        _messageLabel.textAlignment = NSTextAlignmentCenter;
        _messageLabel.hidden = YES;
    }
    return _messageLabel;
}

@end
